<?php
/**
 * Ninja Saga AMF Gateway dengan SabreAMF
 * 
 * Install: composer require sabre/amf
 * Usage: Rename index.php ke index_old.php, rename ini ke index.php
 */

require_once 'vendor/autoload.php';
require_once 'config.php';
require_once 'db.php';

use SabreAMF\AMF;
use SabreAMF\AMF3\Deserializer;
use SabreAMF\AMF3\Serializer;

// Set headers
header('Content-Type: application/x-amf');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Handle preflight
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Include service classes
require_once 'services/SystemService.php';
require_once 'services/CharacterDAO.php';
require_once 'services/SystemData.php';

// AMF Handler
class AMFHandler {
    private $db;
    
    public function __construct($db) {
        $this->db = $db;
    }
    
    public function handle($service, $method, $params) {
        $serviceClass = $this->getServiceClass($service);
        
        if (!$serviceClass) {
            throw new Exception("Service not found: $service");
        }
        
        if (!method_exists($serviceClass, $method)) {
            throw new Exception("Method not found: $service.$method");
        }
        
        return call_user_func_array([$serviceClass, $method], $params);
    }
    
    private function getServiceClass($service) {
        $services = [
            'SystemService' => 'SystemService',
            'CharacterDAO' => 'CharacterDAO',
            'SystemData' => 'SystemData',
        ];
        
        if (!isset($services[$service])) {
            return null;
        }
        
        $className = $services[$service];
        return new $className($this->db);
    }
}

// Main handler
try {
    $db = getDBConnection();
    $handler = new AMFHandler($db);
    
    // Initialize AMF
    $amf = new AMF();
    $amf->setDeserializer(new Deserializer());
    $amf->setSerializer(new Serializer());
    
    // Deserialize AMF request
    $input = file_get_contents('php://input');
    
    try {
        $request = $amf->deserialize($input);
        
        // Parse request - AMF format
        // SabreAMF returns Message object with target, operation, body
        if (is_object($request)) {
            $service = isset($request->target) ? $request->target : 'SystemService';
            $method = isset($request->operation) ? $request->operation : 'login';
            $params = isset($request->body) && is_array($request->body) ? $request->body : [];
        } else {
            // Fallback
            $service = 'SystemService';
            $method = 'login';
            $params = ['test', 'test123', BUILD_NO];
        }
    } catch (Exception $e) {
        // Fallback jika deserialize gagal
        error_log("AMF Deserialize Error: " . $e->getMessage());
        $service = 'SystemService';
        $method = 'login';
        $params = ['test', 'test123', BUILD_NO];
    }
    
    // Handle request
    $result = $handler->handle($service, $method, $params);
    
    // Format response sesuai format Ninja Saga
    // Client mengharapkan object dengan: status, result, signature
    $response = new stdClass();
    $response->status = '1';
    $response->result = $result;
    $response->signature = md5(json_encode($result) . APP_SECRET);
    $response->account_lock = null;
    $response->country_area = null;
    $response->isNewAccount = null;
    $response->promote_id = null;
    $response->isTrialEmblem = 0;
    $response->isExpired = 0;
    
    // Serialize and return AMF response
    // SabreAMF expects to serialize the response object directly
    header('Content-Type: application/x-amf');
    try {
        $amfResponse = $amf->serialize($response);
        echo $amfResponse;
    } catch (Exception $e) {
        error_log("AMF Serialize Error: " . $e->getMessage());
        // Fallback to JSON if AMF serialize fails
        header('Content-Type: application/json');
        echo json_encode($response);
    }
    
} catch (Exception $e) {
    // Error response
    $errorResponse = [
        'status' => '0',
        'error' => $e->getMessage()
    ];
    
    try {
        $amf = new AMF();
        $amf->setSerializer(new Serializer());
        $amfError = $amf->serialize($errorResponse);
        echo $amfError;
    } catch (Exception $e2) {
        // Fallback to JSON if AMF fails
        header('Content-Type: application/json');
        echo json_encode($errorResponse);
    }
}
?>

